import React from "react";
import { useLanguage } from "@/contexts/LanguageContext";
import { TabId } from "@/types/analysis";
import {
  BarChart3,
  DollarSign,
  Zap,
  Building2,
  TrendingUp,
} from "lucide-react";

type Tab = {
  id: TabId;
  name: string;
  icon: React.ComponentType<{ className?: string }>;
};

interface TabNavigationProps {
  activeTab: TabId;
  onTabChange: (tabId: TabId) => void;
}

const tabs: Tab[] = [
  { id: "overview", name: "Overview", icon: BarChart3 },
  { id: "costs", name: "Costs & Payments", icon: DollarSign },
  { id: "performance", name: "Performance", icon: Zap },
  { id: "departments", name: "Departments", icon: Building2 },
  { id: "kpi", name: "KPI Monitoring", icon: TrendingUp },
];

export default function TabNavigation({
  activeTab,
  onTabChange,
}: TabNavigationProps) {
  const { t } = useLanguage();

  const tabs: Tab[] = [
    {
      id: "overview",
      name: t("analysis.tabs.overview") || "Overview",
      icon: BarChart3,
    },
    {
      id: "costs",
      name: t("analysis.tabs.costs") || "Costs & Payments",
      icon: DollarSign,
    },
    {
      id: "performance",
      name: t("analysis.tabs.performance") || "Performance",
      icon: Zap,
    },
    {
      id: "departments",
      name: t("analysis.tabs.departments") || "Departments",
      icon: Building2,
    },
    {
      id: "kpi",
      name: t("analysis.tabs.kpi") || "KPI Monitoring",
      icon: TrendingUp,
    },
  ];

  return (
    <div className='mb-6 sm:mb-8'>
      <div className='border-b border-gray-200 overflow-x-auto overflow-y-hidden'>
        <nav className='flex space-x-2 sm:space-x-4 md:space-x-8 -mb-px min-w-fit'>
          {tabs.map((tab) => (
            <button
              key={tab.id}
              onClick={() => onTabChange(tab.id)}
              className={`py-2 px-2 sm:px-3 md:px-1 border-b-2 font-medium text-xs sm:text-sm flex items-center whitespace-nowrap flex-shrink-0 ${
                activeTab === tab.id
                  ? "border-blue-500 text-blue-600"
                  : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
              }`}
            >
              <tab.icon className='w-3 h-3 sm:w-4 sm:h-4 mr-1 sm:mr-2 flex-shrink-0' />
              <span className='hidden sm:inline'>{tab.name}</span>
              <span className='sm:hidden'>
                {tab.id === "overview"
                  ? t("analysis.tabs.overviewShort") || "Overview"
                  : tab.id === "costs"
                  ? t("analysis.tabs.costsShort") || "Costs"
                  : tab.id === "performance"
                  ? t("analysis.tabs.performanceShort") || "Perf"
                  : tab.id === "departments"
                  ? t("analysis.tabs.departmentsShort") || "Depts"
                  : t("analysis.tabs.kpiShort") || "KPI"}
              </span>
            </button>
          ))}
        </nav>
      </div>
    </div>
  );
}
